package furny.ga.logger;

import furny.ga.FurnEntry;
import furny.ga.FurnLayoutIndividual;
import furny.ga.logger.entities.EvaluationRunEntry;
import furny.ga.logger.entities.EventType;
import furny.ga.logger.entities.GeneEntry;
import furny.ga.logger.entities.IndividualEntry;
import furny.ga.logger.entities.LogEvent;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IPopulation;
import ga.core.logging.IGALogger;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * GA logger that outputs informations to a database.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class GADatabaseLogger implements IGALogger<FurnLayoutIndividual> {
  private EvaluationRunEntry run;

  private final ExecutorService executor = Executors.newFixedThreadPool(4);

  /**
   * Creates a new database logger.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public GADatabaseLogger() {
  }

  @Override
  public void evaluationStarted() {
    run = new EvaluationRunEntry();
    run.setStarted(new Date());

    // save run
    StatisticsDBManager.getInstance().saveEvaluationRun(run, false);
  }

  @Override
  public void allIndividualsEvaluated(final int generation,
      final IPopulation<FurnLayoutIndividual> population) {
    // TODO Auto-generated method stub

  }

  @Override
  public void populationInitiated(final int generation,
      final IPopulation<FurnLayoutIndividual> population) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.POPULATION_INITIATED);

    final List<IndividualEntry> list = new ArrayList<IndividualEntry>();

    for (final FurnLayoutIndividual ind : population) {
      final Set<GeneEntry> genes = new HashSet<GeneEntry>();
      for (final FurnEntry entry : ind.getFurnitures()) {
        genes
            .add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
      }

      final IndividualEntry indEntry = new IndividualEntry(generation, event,
          genes, ind.getFitness());

      list.add(indEntry);
    }

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        for (final IndividualEntry indEntry : list) {
          // save individual entries
          StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
        }
      }
    });
  }

  @Override
  public void individualsSelected(final int generation,
      final IndividualList<FurnLayoutIndividual> individuals) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.INDIVIDUALS_SELECTED);

    final List<IndividualEntry> list = new ArrayList<IndividualEntry>();

    for (final FurnLayoutIndividual ind : individuals) {
      final Set<GeneEntry> genes = new HashSet<GeneEntry>();
      for (final FurnEntry entry : ind.getFurnitures()) {
        genes
            .add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
      }

      final IndividualEntry indEntry = new IndividualEntry(generation, event,
          genes, ind.getFitness());

      list.add(indEntry);
    }

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        for (final IndividualEntry indEntry : list) {
          // save individual entries
          StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
        }
      }
    });
  }

  @Override
  public void individualsCrossed(final int generation,
      final IndividualList<FurnLayoutIndividual> individuals) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.INDIVIDUALS_CROSSED);

    final List<IndividualEntry> list = new ArrayList<IndividualEntry>();

    for (final FurnLayoutIndividual ind : individuals) {
      final Set<GeneEntry> genes = new HashSet<GeneEntry>();
      for (final FurnEntry entry : ind.getFurnitures()) {
        genes
            .add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
      }

      final IndividualEntry indEntry = new IndividualEntry(generation, event,
          genes, ind.getFitness());

      list.add(indEntry);
    }

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        for (final IndividualEntry indEntry : list) {
          // save individual entries
          StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
        }
      }
    });
  }

  @Override
  public void individualsMutated(final int generation,
      final IndividualList<FurnLayoutIndividual> individuals) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.INDIVIDUALS_MUTATED);

    final List<IndividualEntry> list = new ArrayList<IndividualEntry>();

    for (final FurnLayoutIndividual ind : individuals) {
      final Set<GeneEntry> genes = new HashSet<GeneEntry>();
      for (final FurnEntry entry : ind.getFurnitures()) {
        genes
            .add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
      }

      final IndividualEntry indEntry = new IndividualEntry(generation, event,
          genes, ind.getFitness());

      list.add(indEntry);
    }

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        for (final IndividualEntry indEntry : list) {
          // save individual entries
          StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
        }
      }
    });
  }

  @Override
  public void individualsInserted(final int generation,
      final IndividualList<FurnLayoutIndividual> individuals,
      final IPopulation<FurnLayoutIndividual> population) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.INDIVIDUALS_INSERTED);

    final List<IndividualEntry> list = new ArrayList<IndividualEntry>();

    for (final FurnLayoutIndividual ind : population) {
      final Set<GeneEntry> genes = new HashSet<GeneEntry>();
      for (final FurnEntry entry : ind.getFurnitures()) {
        genes
            .add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
      }

      final IndividualEntry indEntry = new IndividualEntry(generation, event,
          genes, ind.getFitness());

      list.add(indEntry);
    }

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        for (final IndividualEntry indEntry : list) {
          // save individual entries
          StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
        }
      }
    });
  }

  @Override
  public void individualSelectedForEvaluation(final int generation,
      final FurnLayoutIndividual individual) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.INDIVIDUALS_SELECTED_FOR_EVALUATION);

    final Set<GeneEntry> genes = new HashSet<GeneEntry>();
    for (final FurnEntry entry : individual.getFurnitures()) {
      genes.add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
    }

    final IndividualEntry indEntry = new IndividualEntry(generation, event,
        genes, individual.getFitness());

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        // save individual entry
        StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
      }
    });
  }

  @Override
  public void individualEvaluated(final int generation,
      final FurnLayoutIndividual individual) {
    final LogEvent event = new LogEvent(new Date(), run,
        EventType.INDIVIDUAL_EVALUATED);

    final Set<GeneEntry> genes = new HashSet<GeneEntry>();
    for (final FurnEntry entry : individual.getFurnitures()) {
      genes.add(new GeneEntry(entry.getVector(), entry.getFurniture().getId()));
    }

    final IndividualEntry indEntry = new IndividualEntry(generation, event,
        genes, individual.getFitness());

    executor.execute(new Runnable() {
      @Override
      public void run() {
        // save event
        StatisticsDBManager.getInstance().saveLogEvent(event);

        // save individual entry
        StatisticsDBManager.getInstance().saveIndividualEntry(indEntry);
      }
    });
  }

  @Override
  public void exit() {
    run.setEnded(new Date());

    // save run
    StatisticsDBManager.getInstance().saveEvaluationRun(run, true);

    executor.shutdown();

    StatisticsDBManager.getInstance().exit();

    System.out.println("logger exited");
  }
}
